function [cubes,scubes] = syn_imalign(ori,flprot_idx,varargin)
%This function uses the rotation index generated by syn_align, and rotates
%the actual image data cubes.
%Synatax:   [cubes] = syn_imalign(ori,flprot_idx,'summerize',1,'type',1);
%Input:     ori = a list of vertices.  Note: if an array have more than
%               3 columns, only the first three will be considered. Do not
%               usre the ori output from syn_align, use the original
%               vertices list.
%           flprot_idx = the flip rotation index generated by syn_align
%           'summarize' = Takes all of rotated cubes and create a summary
%               set of cubes. 1 = on, 0 = off(default).  Also, when summary
%               is off, the summarized cubes will be output as image files
%           'type' = How the summary is performed. 1 = median(default), 2 =
%               mean, 3 = max, 4 = min, 5 = standard deviation, 6 =
%               variance
%Output:    cubes = the image cubes.  If summary is off, this is all of the
%               rotated cubes for all of the channels.  If summary is on
%               this is the set of summarized cubes.
%           scubes = the summarized cubes.  If summary is off, ave_cubes
%               is empty.

%first grab the image stacks
[stks,img_range,filenames,pathname] = stack_gui(1);  %use cell output mode.

[summarize,type] = parse(varargin);    %parse input

%now generate the cubes
cubes = cubed(stks,ori,'round',1,'size',10);

%generate some affine matrices.
flpxm = [-1 0 0 0;0 -1 0 0;0 0 1 0;0 0 0 1];    %flip 180deg in x y plane (rotate about z)

%now rotate the cubes
x_max = 0;  %initiate
y_max = 0;
z_max = 0;
for i = 1:size(cubes,2)     %flip through the channels
    for j = 1:size(cubes,1)     %dash through the individual cubes in each channel.
        %first check for flipping, work on the xy plane first.
        if flprot_idx(j,1)  %yes x flip
            cubes{j,i} = affine(cubes{j,i},flpxm,[],0,0);   %rotate, off verbose, background = 0
        end
        %now create the two rotation matices
        rotxm = [cosd(flprot_idx(j,2)) -sind(flprot_idx(j,2)) 0 0;sind(flprot_idx(j,2)) cosd(flprot_idx(j,2)) 0 0;...
            0 0 1 0;0 0 0 1];       %rotate around the z towards x in the xy plane
        rotzm = [cosd(flprot_idx(j,3)) 0 sind(flprot_idx(j,3)) 0;0 1 0 0;-sind(flprot_idx(j,3)) 0 cosd(flprot_idx(j,3)) 0;...
            0 0 0 1];       %rotate around the y towards x in the xz plane
        %now rotate the matrices
        cubes{j,i} = affine(cubes{j,i},rotxm,[],0,0);     %rotate in the xy
        cubes{j,i} = affine(cubes{j,i},rotzm,[],0,0);     %rotate in the zy
        %get a running idea of the max cube size 
        [y_tmp,x_tmp,z_tmp] = size(cubes{j,i});
        if x_tmp>x_max
            x_max = x_tmp;
        end
        if y_tmp>y_max
            y_max = y_tmp;
        end
        if z_tmp>z_max
            z_max = z_tmp;
        end
    end
end

if summarize        %average the cubes
    %we will need to pad the cubes so they are the same size
    for k = 1:size(cubes,2)     %go throught the channels again
        all_cube = zeros(x_max,y_max,z_max,size(cubes,1));      %preallocate
        for l = 1:size(cubes,1)     %iterate through the individual cubes
            %pad each dimension individually
            if y_max>size(cubes{l,k},1)     %pad y dimension, aka rows
                pad_tmp = y_max-size(cubes{l,k},1);    %how much to pad
                if rem(pad_tmp,2)   %if there is a remainder pad off center
                    cubes{l,k} = padarray(cubes{l,k},[floor(pad_tmp/2) 0 0]);   %pad evenly first
                    cubes{l,k} = padarray(cubes{l,k},[1 0 0],'pre');    %one row of extra padding along the front
                else    %even good, ideal, pad around
                    cubes{l,k} = padarray(cubes{l,k},[pad_tmp/2 0 0]);
                end
            end
            if x_max>size(cubes{l,k},2)     %pad x dimension, aka col
                pad_tmp = x_max-size(cubes{l,k},2);    %how much to pad
                if rem(pad_tmp,2)   %if there is a remainder pad off center
                    cubes{l,k} = padarray(cubes{l,k},[0 floor(pad_tmp/2) 0]);   %pad evenly first
                    cubes{l,k} = padarray(cubes{l,k},[0 1 0],'pre');    %one row of extra padding along the front
                else    %even good, ideal, pad around
                    cubes{l,k} = padarray(cubes{l,k},[0 pad_tmp/2 0]);
                end
            end
            if z_max>size(cubes{l,k},3)     %pad z dimension, aka umm 3rd dimension
                pad_tmp = x_max-size(cubes{l,k},3);    %how much to pad
                if rem(pad_tmp,2)   %if there is a remainder pad off center
                    cubes{l,k} = padarray(cubes{l,k},[0 0 floor(pad_tmp/2)]);   %pad evenly first
                    cubes{l,k} = padarray(cubes{l,k},[0 0 1],'pre');    %one row of extra padding along the front
                else    %even good, ideal, pad around
                    cubes{l,k} = padarray(cubes{l,k},[0 0 pad_tmp/2]);
                end
            end
            %now cat the cubes together into fourth dimension for quick
            %math.
            all_cube(:,:,:,l) = cubes{l,k};
        end
        %now to make the summary cubes
        switch type
            case 'median'
                scube{1,k} = median(all_cube,4);     %the median value of all cubes for this channel
            case 'mean'
                scube{1,k} = mean(all_cube,4);       %the mean
            case 'std'
                scube{1,k} = std(all_cube,0,4);      %the standard devidation
            case 'var'
                scube{1,k} = var(all_cube,0,4);      %the variance
            case 'max'
                scube{1,k} = max(all_cube,[],4);     %the max projection in 3d
            case 'min'
                scube{1,k} = median(all_cube,[],4);  %the min projection in 3d
        end
        %save the average cubes out
        %make the output directory
        mkdir(pathname,['summary-',type]);
        %localization for macs
        if ispc
            slash = '\';        %Windows directory marker
        else
            slash = '/';        %Mac directory marker
        end
        %Now save out the image files
        mkdir([pathname,['summary-',type]],filenames{1,k}(1:end-4));
        stk2tiff(scube{1,k},filenames{1,k},[pathname,['summary-',type],slash,filenames{1,k}(1:end-4),slash]);
    end
else
    scubes = [];     %no summary.
end
%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [summarize,type] = parse(input)
        
summarize = 0;  %Default Initialized.
type = 'median';

%Parse the input
if ~isempty(input)
    for i = 1:2:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'summarize'
                    summarize = input{1,i+1};
                case 'type'
                    type = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        end
    end
end